# Plug-in SDK Documentation

## Contents

- [Shader and Filter Plugins](#shader-and-filter-plugins)
- [JavaScript Utility Plugins](#javascript-plugins)
- [JavaScript Shape Plugins](#javascript-shape-plugins)
- [JavaScript Deformer Plugins](#javascript-deformer-plugins)

---

## Shader and Filter Plugins

### Anatomy of a Shader or Filter Plug-in

Third-party Shader and Filter plugins consist of several files in a folder:

```
📁 Layer
  ┠ definitions.json
  ┠ layer.sksl
  ┠ layerIcon.png
  ┠ layerIcon@2x.png
  ┠ setup.js
  ┠ strings.json
  ┠ welcome.js
  ┠ versioning.js
```

---

### Required Files

#### Layer SkSL File

The SkSL code defining the Layer. Can be encrypted using `encryptFile` from Cavalry's API Module (use `.skslc` extension for encrypted files).

#### Definitions JSON File

Contains attribute definitions, UI order, icon name, and Layer metadata. Multiple Layers can be defined in the same file.

**Author Key**: Both `definitions.json` and `strings.json` must include an `"author"` key as an identifier (e.g. company name). Must not contain spaces or periods; preferably camelCase.

```json
"author": "sceneGroup",
```

**SuperTypes**:
- Filter: `"thirdPartyFilter"`
- Shader: `"thirdPartyShader"`

#### Strings JSON File

Contains nice names and tooltips for attributes. Must include the author key.

---

### Optional Files

#### setup.js

JavaScript run when creating a Layer. Access the layer ID via `setup.layerId`.

#### welcome.js

Splash screen shown on installation. Use Script UI to create the dialogue. Check `install.fromUpdate` for update vs fresh install.

#### versioning.js

Run when opening Scenes containing the plugin's Layers. Access `plugin.layerId` and `plugin.saveData` for backwards compatibility.

---

### Icons

| Icon Type | Dimensions | Suffix | Example |
|-----------|------------|--------|---------|
| Layer | 18×18px | — | `layerIcon.png` |
| Layer (Retina) | 36×36px | `@2x` | `layerIcon@2x.png` |
| Attribute Editor | 16×16px | `_ae` | `layerIcon_ae.png` |
| Attribute Editor (Retina) | 32×32px | `_ae@2x` | `layerIcon_ae@2x.png` |

Native Shaders and Filters use colours `#F5B4B8` and `#CACACA`.

---

### Minimal Templates

#### Minimal definitions.json

```json
[{
  "author": "yourName",
  "type": "myFilter",
  "superType": "thirdPartyFilter",
  "version": "1.0",
  "skslFile": "myFilter.sksl",
  "attributes": {
    "amount": { "type": "double", "default": 1.0 }
  },
  "UI": {
    "icon": "myFilter.png"
  }
}]
```

#### Minimal strings.json

```json
[{
  "type": "layerStrings",
  "value": {
    "author": "yourName",
    "layerType": "myFilter",
    "niceName": "My Filter",
    "layerInfo": "A description of what this filter does.",
    "language": "en",
    "attributes": {
      "amount": ["Amount", "Controls the effect intensity"]
    }
  }
}]
```

#### Minimal Filter SkSL

```glsl
uniform float2 resolution;
uniform shader childShader;
uniform float amount;

float4 main(float2 coord) {
    float4 col = childShader.eval(coord);
    // Your filter logic here
    return col;
}
```

#### Minimal Shader SkSL

```glsl
uniform float2 resolution;
uniform float amount;

float4 main(float2 coord) {
    // Your shader logic here
    return float4(coord.x / resolution.x, coord.y / resolution.y, 0.0, 1.0);
}
```

---

## JavaScript Plugins

JavaScript plugins use JavaScript to create custom layers. There are three types:

| Type | SuperType | Use Case |
|------|-----------|----------|
| Utility | `thirdPartyJavaScript` | Generate values (numbers, colours, strings) |
| Shape | `thirdPartyJavaScriptShape` | Create custom shapes |
| Deformer | `thirdPartyJavaScriptDeformer` | Modify existing shapes |

---

### Common Files (All JavaScript Plugins)

All JavaScript plugins share the same file structure:

```
📁 MyPlugin
  ┠ definitions.json     (required)
  ┠ strings.json         (required)
  ┠ myPlugin.js          (required, or .jsc for encrypted)
  ┠ layerIcon.png        (optional)
  ┠ layerIcon@2x.png     (optional)
  ┠ setup.js             (optional)
  ┠ welcome.js           (optional)
  ┠ versioning.js        (optional)
```

#### definitions.json

Contains attribute definitions, UI order, icon name, and layer metadata. Must include the `"author"` key.

#### strings.json

Contains nice names and tooltips for attributes. Must include the `"author"` key matching definitions.json.

#### JavaScript File

The main JavaScript file containing your plugin logic. Can be encrypted using `cavalry.encryptFile()` (use `.jsc` extension for encrypted files).

**Input Values**: Input attributes are available as JavaScript variables using their attribute names. For example, an attribute named `"amount"` is accessed as `amount` in the script.

#### Optional Files

| File | Purpose |
|------|---------|
| setup.js | Runs when creating a layer. Access the layer ID via `setup.layerId`. |
| welcome.js | Splash screen shown on installation. Check `install.fromUpdate` for update vs fresh install. |
| versioning.js | Runs when opening scenes containing the plugin's layers. Access `plugin.layerId` and `plugin.saveData`. |

#### Icons

| Icon Type | Dimensions | Suffix | Example |
|-----------|------------|--------|---------|
| Layer | 18×18px | — | `layerIcon.png` |
| Layer (Retina) | 36×36px | `@2x` | `layerIcon@2x.png` |
| Attribute Editor | 16×16px | `_ae` | `layerIcon_ae.png` |
| Attribute Editor (Retina) | 32×32px | `_ae@2x` | `layerIcon_ae@2x.png` |

---

### JavaScript Utility Plugins

Utility plugins generate output values that can be connected to other layer attributes. The result of the last expression in the script becomes the output value.

#### Supported Output Types

- `double`, `double2`, `double3`
- `int`, `int2`, `int3`
- `bool`
- `string`, `richText`
- `color`
- `polyMesh`
- `pointData`

Set the output type in the `supportedOutputTypes` array of the `out` attribute in definitions.json.

#### Minimal definitions.json

```json
[{
  "author": "yourName",
  "type": "myUtility",
  "superType": "thirdPartyJavaScript",
  "jsFile": "myUtility.js",
  "version": "1.0",
  "attributes": {
    "time": { "type": "double", "default": 0.0, "compConnect": "time" },
    "frequency": { "type": "double", "default": 0.1, "min": 0.001 },
    "amplitude": { "type": "double", "default": 100.0 }
  },
  "triggers": {
    "out": ["time", "frequency", "amplitude"]
  },
  "UI": {
    "attributeOrder": ["time", "frequency", "amplitude"],
    "icon": "myUtility.png"
  }
}]
```

#### Minimal strings.json

```json
[{
  "type": "layerStrings",
  "value": {
    "author": "yourName",
    "layerType": "myUtility",
    "niceName": "My Utility",
    "layerInfo": "Generates a sine wave based on time input.",
    "language": "en",
    "attributes": {
      "time": ["Time", "The input time value."],
      "frequency": ["Frequency", "Wave frequency in cycles per frame."],
      "amplitude": ["Amplitude", "Wave amplitude."]
    }
  }
}]
```

#### Minimal JavaScript File

```javascript
Math.sin(time * frequency * Math.PI * 2) * amplitude;
```

---

### JavaScript Shape Plugins

Shape plugins create custom shapes. They automatically have access to transforms, materials (fill/stroke), deformers, filters, and masks.

The JavaScript file must return a `cavalry.Mesh` or `cavalry.Path` object.

#### Minimal definitions.json

```json
[{
  "author": "yourName",
  "type": "myShape",
  "superType": "thirdPartyJavaScriptShape",
  "jsFile": "myShape.js",
  "version": "1.0",
  "attributes": {
    "radius": { "type": "double", "default": 100.0, "min": 0.0 },
    "segments": { "type": "int", "default": 64, "min": 3 }
  },
  "triggers": {
    "polyMesh": ["radius", "segments"]
  },
  "UI": {
    "attributeOrder": ["radius", "segments"],
    "icon": "myShape.png"
  }
}]
```

#### Minimal strings.json

```json
[{
  "type": "layerStrings",
  "value": {
    "author": "yourName",
    "layerType": "myShape",
    "niceName": "My Shape",
    "layerInfo": "A custom shape layer.",
    "language": "en",
    "attributes": {
      "radius": ["Radius", "The radius of the shape."],
      "segments": ["Segments", "Number of segments."]
    }
  }
}]
```

#### Minimal JavaScript File

```javascript
(function() {
    var path = new cavalry.Path();
    path.addEllipse(0, 0, radius, radius);

    var mesh = new cavalry.Mesh();
    mesh.addPath(path);
    return mesh;
})();
```

#### Materials

When adding paths to a mesh, the material argument is optional:

```javascript
mesh.addPath(path);                        // No material - uses Fill/Stroke from Cavalry UI
mesh.addPath(path, new cavalry.Material()); // Explicit material defined in code
```

Omitting the material is typically preferred - this allows users to control fill and stroke via the standard Cavalry UI (Fill/Stroke tabs). Only specify a material in code when you need programmatic control over colours or stroke settings.

See the `cavalry` module documentation for the full Path, Mesh, and Material API.

---

### JavaScript Deformer Plugins

Deformer plugins modify existing shapes. They have access to the `def` module for manipulating meshes and support falloffs for localised effects.

#### Minimal definitions.json

```json
[{
  "author": "yourName",
  "type": "myDeformer",
  "superType": "thirdPartyJavaScriptDeformer",
  "jsFile": "myDeformer.js",
  "version": "1.0",
  "attributes": {
    "amount": { "type": "double", "default": 10.0 }
  },
  "triggers": {
    "out": ["amount"]
  },
  "UI": {
    "attributeOrder": ["amount"]
  }
}]
```

#### Minimal strings.json

```json
[{
  "type": "layerStrings",
  "value": {
    "author": "yourName",
    "layerType": "myDeformer",
    "niceName": "My Deformer",
    "layerInfo": "A custom deformer layer.",
    "language": "en",
    "attributes": {
      "amount": ["Amount", "The deformation amount."]
    }
  }
}]
```

#### Minimal JavaScript File

```javascript
function visitAllMeshes(mesh, func) {
    for (var i = 0; i < mesh.childMeshCount(); i++) {
        var child = mesh.getChildMeshAtIndex(i);
        visitAllMeshes(child, func);
        mesh.setChildMeshAtIndex(i, child);
    }
    func(mesh);
}

function deformMesh(mesh) {
    var pathCount = mesh.count();
    for (var p = 0; p < pathCount; p++) {
        var path = mesh.getPathAtIndex(p);
        var pd = path.pathData();

        for (var idx = 0; idx < pd.length; idx++) {
            var pt = pd[idx].point;
            var normal = pd[idx].normal;
            var falloff = def.getFalloffAtPoint(pt.x, pt.y);

            pd[idx].point.x += normal.x * amount * falloff;
            pd[idx].point.y += normal.y * amount * falloff;
        }

        path.setPathData(pd);
        mesh.setPathAtIndex(p, path);
    }
}

var root = def.getRootMesh();
visitAllMeshes(root, deformMesh);
def.setRootMesh(root);
```

See the `def` module documentation for the full deformer API reference.

---

## Installation

Drag the plugin folder (or a .zip of it) anywhere into the Cavalry window. A confirmation dialogue appears if valid.

**Installation paths**:
- **macOS**: `~/Library/Application Support/Cavalry/Third-Party/Plugins`
- **Windows**: `C:\Users\<USER>\AppData\Roaming\Cavalry\Third-Party\Plugins`

> **Note**: These are hidden folders. On Windows, enable "Hidden Items" in Explorer. On macOS, hold Option/Alt when opening Finder's Go menu.

---

## Next Steps

See the example folders for complete working plugins:
- `Examples/Filters/` - Single-pass filter examples
- `Examples/Shaders/` - Single-pass shader examples
- `Examples/Multi-Pass Filters/` - Multi-pass filter example
- `Examples/Multi-Pass Shaders/` - Multi-pass shader example
- `Examples/JavaScript/Sin Wave Generator/` - JavaScript utility plugin example
- `Examples/JavaScript/Trefoil Shape/` - JavaScript shape plugin example
- `Examples/JavaScript/Greeble/` - JavaScript deformer plugin example with falloff support
- `Examples/JavaScript/Spikes/` - JavaScript deformer plugin example with falloff support

Refer to other documentation for specific topics:

**Shaders and Filters:**
- `Shaders and Filters/SkSL Tips.md` - SkSL limitations and best practices
- `Shaders and Filters/Coordinate System.md` - Coordinate system details
- `Shaders and Filters/Multipass Shaders and Filters.md` - Multi-pass configuration
- `Shaders and Filters/Shader Generator Inputs for Filters.md` - Using shaders as filter inputs

**JavaScript:**
- `JavaScript/JavaScript Plugin Tips.md` - Expression returns, complex logic patterns, and more

**Schemas:**
- `definitionsSchema.json` - Full definitions schema reference
- `stringsSchema.json` - Full strings schema reference

---